/*
 * Copyright 2023 KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <QDebug>

#include "upowerbatterydbus.h"

UpowerBatteryDBus::UpowerBatteryDBus(int batteryId, const QString &batteryDBusPath, QObject *parent)
    : QObject(parent), m_batteryId(batteryId)
{
    m_DBusInterface = new QDBusInterface("org.freedesktop.UPower",
                                         batteryDBusPath,
                                         "org.freedesktop.DBus.Properties",
                                         QDBusConnection::systemBus(),
                                         this);

    QDBusConnection::systemBus().connect("org.freedesktop.UPower",
                                         batteryDBusPath,
                                         "org.freedesktop.DBus.Properties",
                                         "PropertiesChanged",
                                         this,
                                         SLOT(dealBatteryPropertiesMessage(QDBusMessage)));

    initBatteryInfo();
}

UpowerBatteryDBus::~UpowerBatteryDBus() {}

void UpowerBatteryDBus::dealBatteryPropertiesMessage(QDBusMessage msg)
{
    const QDBusArgument &dbusArgs = msg.arguments().at(1).value<QDBusArgument>();
    QMap<QString, QVariant> map;
    dbusArgs >> map;

    QStringList batteryInfoChangedList;

    if (map.contains("State")) {
        m_batteryState = map.value(QString("State")).toInt();
        batteryInfoChangedList.append("State");
    }
    if (map.contains("Percentage")) {
        m_percentage = map.value(QString("Percentage")).toDouble();
        batteryInfoChangedList.append("Percentage");
    }
    if (map.contains("Energy")) {
        m_energy = map.value(QString("Energy")).toDouble();
        batteryInfoChangedList.append("Energy");
    }
    if (map.contains("EnergyRate")) {
        m_energyRate = map.value(QString("EnergyRate")).toDouble();
        batteryInfoChangedList.append("EnergyRate");
    }
    if (map.contains("TimeToFull")) {
        m_timeToFull = map.value(QString("TimeToFull")).toLongLong();
        batteryInfoChangedList.append("TimeToFull");
    }
    if (map.contains("TimeToEmpty")) {
        m_timeToEmpty = map.value(QString("TimeToEmpty")).toLongLong();
        batteryInfoChangedList.append("TimeToEmpty");
    }
    if (map.contains("Temperature")) {
        m_temperature = map.value(QString("Temperature")).toDouble();
        batteryInfoChangedList.append("Temperature");
    }
    if (map.contains("WarningLevel")) {
        m_warninglevel = map.value(QString("WarningLevel")).toInt();
        batteryInfoChangedList.append("WarningLevel");
    }

    emit batteryInfoChanged(m_batteryId, batteryInfoChangedList);
}

void UpowerBatteryDBus::initBatteryInfo()
{
    QDBusMessage msg = m_DBusInterface->call("GetAll", "org.freedesktop.UPower.Device");

    if (msg.type() == QDBusMessage::ReplyMessage) {
        const QDBusArgument &dbusArgs = msg.arguments().at(0).value<QDBusArgument>();
        QMap<QString, QVariant> map;
        dbusArgs >> map;

        m_batteryState = map.value(QString("State")).toInt();
        m_percentage = map.value(QString("Percentage")).toDouble();
        m_energy = map.value(QString("Energy")).toDouble();
        m_energyFull = map.value(QString("EnergyFull")).toDouble();
        m_energyFullDesign = map.value(QString("EnergyFullDesign")).toDouble();
        m_energyRate = map.value(QString("EnergyRate")).toDouble();
        m_timeToFull = map.value(QString("TimeToFull")).toLongLong();
        m_timeToEmpty = map.value(QString("TimeToEmpty")).toLongLong();
        m_temperature = map.value(QString("Temperature")).toDouble();
        m_warninglevel = map.value(QString("WarningLevel")).toInt();
        m_vendor = map.value(QString("Vendor")).toString();
        m_model = map.value(QString("Model")).toString();
    }
}

int UpowerBatteryDBus::getBatteryState()
{
    return m_batteryState;
}

double UpowerBatteryDBus::getBatteryPercentage()
{
    return m_percentage;
}

double UpowerBatteryDBus::getEnergyNow()
{
    return m_energy;
}

double UpowerBatteryDBus::getEnergyFull()
{
    return m_energyFull;
}

double UpowerBatteryDBus::getEnergyFullDesign()
{
    return m_energyFullDesign;
}

double UpowerBatteryDBus::getEnergyRate()
{
    return m_energyRate;
}

qlonglong UpowerBatteryDBus::getTimeToFull()
{
    return m_timeToFull;
}

qlonglong UpowerBatteryDBus::getTimeToEmpty()
{
    return m_timeToEmpty;
}

double UpowerBatteryDBus::getBatteryTemperature()
{
    return m_temperature;
}

int UpowerBatteryDBus::getWarninglevel()
{
    return m_warninglevel;
}

QString UpowerBatteryDBus::getBatteryVendor()
{
    return m_vendor;
}

QString UpowerBatteryDBus::getBatteryModel()
{
    return m_model;
}
