package prefixfile

import (
	"bytes"
	"encoding/json"
	"io"
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestDecodeJSONSlurm(t *testing.T) {
	json, err := os.Open("slurm.json")
	if err != nil {
		panic(err)
	}
	decoded, err := DecodeJSONSlurm(json)
	if err != nil {
		t.Errorf("Unable to decode json: %v", err)
	}
	assert.Nil(t, err)
	asn, _ := decoded.ValidationOutputFilters.PrefixFilters[1].GetASN()
	_, asnEmpty := decoded.ValidationOutputFilters.PrefixFilters[0].GetASN()
	assert.Equal(t, uint32(64496), asn)
	assert.True(t, asnEmpty)
	assert.Equal(t, "192.0.2.0/24", decoded.ValidationOutputFilters.PrefixFilters[0].Prefix)
}

func TestFilterOnVRPs(t *testing.T) {
	vrps := []VRPJson{
		{
			ASN:    uint32(65001),
			Prefix: "192.168.0.0/25",
			Length: 25,
		},
		{
			ASN:    uint32(65002),
			Prefix: "192.168.1.0/24",
			Length: 24,
		},
		{
			ASN:    uint32(65003),
			Prefix: "192.168.2.0/24",
			Length: 24,
		},
		{
			ASN:    uint32(65004),
			Prefix: "10.0.0.0/24",
			Length: 24,
		},
		{
			ASN:    uint32(65005),
			Prefix: "10.1.0.0/24",
			Length: 16, // this VRP is broken, maxlength can't be smaller than plen
		},
	}

	asA, asB := uint32(65001), uint32(65002)
	slurm := SlurmValidationOutputFilters{
		PrefixFilters: []SlurmPrefixFilter{
			{
				Prefix: "10.0.0.0/8",
			},
			{
				ASN:    &asA,
				Prefix: "192.168.0.0/24",
			},
			{
				ASN: &asB,
			},
		},
	}
	added, removed := slurm.FilterOnVRPs(vrps)
	assert.Len(t, added, 1)
	assert.Len(t, removed, 4)
	assert.Equal(t, uint32(65001), removed[0].GetASN())
	assert.Equal(t, uint32(65005), removed[3].GetASN())
}

func TestAssertVRPs(t *testing.T) {
	slurm := SlurmLocallyAddedAssertions{
		PrefixAssertions: []SlurmPrefixAssertion{
			{
				ASN:     uint32(65001),
				Prefix:  "10.0.0.0/8",
				Comment: "Hello",
			},
			{
				ASN:    uint32(65001),
				Prefix: "192.168.0.0/24",
			},
			{
				ASN:             uint32(65003),
				Prefix:          "192.168.0.0/25",
				MaxPrefixLength: 26,
			},
		},
	}
	vrps := slurm.AssertVRPs()
	assert.Len(t, vrps, 3)
}

func TestFilterOnBSKs(t *testing.T) {
	vrps := []BgpSecKeyJson{
		{
			Asn: 65001,
			Pubkey: []byte{
				0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
				0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x80, 0x57, 0x23, 0x43, 0xf8,
				0x3f, 0xfc, 0xb0, 0x10, 0x7a, 0xb0, 0x07, 0xd8, 0xca, 0x69, 0xf8, 0x6b, 0x9c, 0xa0, 0x30, 0x06,
				0x05, 0xb8, 0x48, 0xa8, 0x3d, 0xf7, 0xc0, 0xd3, 0xec, 0x5f, 0x19, 0xc0, 0x19, 0xbf, 0xa6, 0xb5,
				0x9e, 0xd7, 0x42, 0xb5, 0x4e, 0xf4, 0x34, 0x3a, 0x52, 0x50, 0x12, 0x86, 0xd8, 0xa0, 0xe7, 0xe4,
				0x1f, 0x10, 0xaa, 0x53, 0xb4, 0x58, 0x22, 0xa9, 0xf8, 0x80, 0x15,
			},
			Ski: "5d4250e2d81d4448d8a29efce91d29ff075ec9e2",
		},
		{
			Asn: 65003,
			Pubkey: []byte{
				0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
				0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xe0, 0x5c, 0x49, 0xaf, 0x49,
				0xf6, 0x6e, 0xec, 0x75, 0xb9, 0x7d, 0x44, 0xbe, 0x5f, 0x90, 0x5b, 0x06, 0x58, 0xbc, 0x86, 0x9d,
				0x3e, 0x32, 0xee, 0x15, 0x7d, 0xa6, 0xc6, 0xa2, 0xae, 0x00, 0x65, 0x21, 0x2a, 0x7a, 0xfb, 0x54,
				0xb2, 0xc3, 0x82, 0xb1, 0x3e, 0xfa, 0x5f, 0x69, 0xe5, 0xe1, 0xf6, 0x91, 0x64, 0xcd, 0x54, 0x03,
				0x76, 0xd8, 0x55, 0x14, 0xdd, 0xd6, 0xff, 0x44, 0xaa, 0x44, 0xdb,
			},
			Ski: "be889b55d0b737397d75c49f485b858fa98ad11f",
		},
		{
			Asn: 65002,
			Pubkey: []byte{
				0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
				0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x86, 0xfe, 0x47, 0x10, 0x11,
				0xa2, 0xc5, 0x48, 0xca, 0x25, 0x39, 0x5e, 0x9e, 0xf7, 0x03, 0xd4, 0x0c, 0x72, 0x8b, 0x4e, 0xeb,
				0x15, 0xd5, 0x58, 0xd4, 0xa8, 0x4d, 0xe2, 0xf3, 0x0f, 0x63, 0x2e, 0x72, 0xd0, 0xcc, 0x7a, 0xcd,
				0xf6, 0xa2, 0x12, 0xa2, 0x4d, 0xdb, 0xb8, 0xca, 0xfe, 0x5e, 0xb5, 0xc4, 0x2d, 0xfa, 0x56, 0xc6,
				0x9e, 0xcd, 0xde, 0xde, 0x5c, 0x0b, 0x19, 0xd4, 0x01, 0x04, 0xb1,
			},
			Ski: "510f485d29a29db7b515f9c478f8ed3cb7aa7d23",
		},
		{
			Asn: 65004,
			Pubkey: []byte{
				0x30, 0x59, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			Ski: "111b485d29a29db7b515f9c471e1ed3cb7bb7dee",
		},
		{
			Asn: 65005,
			Pubkey: []byte{
				0x30, 0x59, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			Ski: "111b485d29a29db7b515f9c471e1ed3cb7bb7dee",
		},
	}

	asA, asB := uint32(65001), uint32(65005)
	slurm := SlurmValidationOutputFilters{
		BgpsecFilters: []SlurmBGPsecFilter{
			{
				SKI: []byte{0xbe, 0x88, 0x9b, 0x55, 0xd0, 0xb7, 0x37, 0x39, 0x7d, 0x75, 0xc4, 0x9f, 0x48, 0x5b, 0x85, 0x8f, 0xa9, 0x8a, 0xd1, 0x1f},
			},
			{
				ASN: &asA,
			},
			{
				SKI: []byte{0x11, 0x1b, 0x48, 0x5d, 0x29, 0xa2, 0x9d, 0xb7, 0xb5, 0x15, 0xf9, 0xc4, 0x71, 0xe1, 0xed, 0x3c, 0xb7, 0xbb, 0x7d, 0xee},
				ASN: &asB,
			},
		},
	}
	added, removed := slurm.FilterOnBRKs(vrps)
	assert.Len(t, added, 2)
	assert.Len(t, removed, 3)
	assert.Equal(t, "5d4250e2d81d4448d8a29efce91d29ff075ec9e2", removed[0].Ski)
	assert.Equal(t, "be889b55d0b737397d75c49f485b858fa98ad11f", removed[1].Ski)
	assert.Equal(t, "111b485d29a29db7b515f9c471e1ed3cb7bb7dee", removed[2].Ski)
	assert.Equal(t, uint32(65005), removed[2].Asn)
}

func TestSlurmEndToEnd(t *testing.T) {
	slurmfd, err := os.Open("slurm.json")
	if err != nil {
		panic(err)
	}
	config, err := DecodeJSONSlurm(slurmfd)
	if err != nil {
		t.Errorf("Unable to decode json: %v", err)
	}

	rpkifd, err := os.Open("test.rpki.json")
	if err != nil {
		panic(err)
	}
	rpkidata, err := io.ReadAll(rpkifd)
	if err != nil {
		panic(err)
	}
	vrplist, err := decodeJSON(rpkidata)
	if err != nil {
		panic(err)
	}

	finalVRP, finalBgpsec := config.FilterAssert(vrplist.ROA, vrplist.BgpSecKeys, nil)

	foundAssertVRP := false
	for _, vrps := range finalVRP {
		if vrps.Prefix == "192.0.2.0/24" {
			t.Fatalf("Found filtered VRP")
		}

		if vrps.Prefix == "198.51.100.0/24" {
			foundAssertVRP = true
		}
	}
	if !foundAssertVRP {
		t.Fatalf("Did not find asserted VRP")
	}

	foundAssertBRK := false
	for _, brks := range finalBgpsec {
		if brks.Ski == "510f485d29a29db7b515f9c478f8ed3cb7aa7d23" {
			t.FailNow()
		}
		if brks.Ski == "3506176743e02f67dd46c73119d5436be7e10106" {
			foundAssertBRK = true
		}
	}
	if !foundAssertBRK {
		t.Fatalf("Did not find asserted BR")
	}
}

func decodeJSON(data []byte) (*RPKIList, error) {
	buf := bytes.NewBuffer(data)
	dec := json.NewDecoder(buf)

	var vrplistjson RPKIList
	err := dec.Decode(&vrplistjson)
	return &vrplistjson, err
}
