% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllClasses.R, R/SpatialExperiment.R
\docType{class}
\name{SpatialExperiment-class}
\alias{SpatialExperiment-class}
\alias{SpatialExperiment}
\title{The SpatialExperiment class}
\arguments{
\item{...}{Arguments passed to the \code{\link{SingleCellExperiment}}
constructor to fill the slots of the base class.}

\item{sample_id}{A \code{character} sample identifier, which matches the
\code{sample_id} in \code{\link{imgData}}. The \code{sample_id} will also
be stored in a new column in \code{\link{colData}}, if not already present.
Default = \code{sample01}.}

\item{spatialCoordsNames}{A \code{character} vector of column names from
\code{\link{colData}} containing spatial coordinates, which will be
accessible with \code{\link{spatialCoords}}. Alternatively, the
\code{spatialCoords} argument may be provided. If both are provided,
\code{spatialCoordsNames} is given precedence, and a warning is returned.
Default = \code{c("x", "y")}.}

\item{spatialCoords}{A numeric matrix containing columns of spatial
coordinates, which will be accessible with \code{\link{spatialCoords}}.
Alternatively, \code{spatialCoordsNames} may be provided. If both are
provided, \code{spatialCoordsNames} is given precedence, and a warning is
returned.}

\item{scaleFactors}{Optional scale factors associated with the image(s). This
can be provided as a numeric value, numeric vector, list, or file path to a
JSON file for the 10x Genomics Visium platform. For 10x Genomics Visium,
the correct scale factor will automatically be selected depending on the
resolution of the image from \code{imageSources}. Default = \code{1}.}

\item{imgData}{Optional \code{\link{DataFrame}} containing the image data.
Alternatively, this can be built from the arguments \code{imageSources} and
\code{image_id} (see Details).}

\item{imageSources}{Optional file path(s) or URL(s) for one or more image
sources.}

\item{image_id}{Optional character vector (same length as
\code{imageSources}) containing unique image identifiers.}

\item{loadImage}{Logical indicating whether to load image into memory.
Default = \code{FALSE}.}

\item{spatialDataNames}{(Deprecated) A \code{character} vector of column
names from \code{\link{colData}} to include in \code{\link{spatialData}}.
Alternatively, the \code{spatialData} argument may be provided. If both are
provided, \code{spatialDataNames} is given precedence, and a warning is
returned. (Note: \code{spatialData} and \code{spatialDataNames} have been
deprecated; \code{colData} and \code{spatialCoords} should be used for all
columns. The arguments have been retained for backward compatibility but
may be removed in the future.)}

\item{spatialData}{(Deprecated) A \code{\link{DataFrame}} containing columns
to store in \code{\link{spatialData}}, which must contain at least the
columns of spatial coordinates. Alternatively, \code{spatialDataNames} may
be provided. If both are provided, \code{spatialDataNames} is given
precedence, and a warning is returned. (Note: \code{spatialData} and
\code{spatialDataNames} have been deprecated; \code{colData} and
\code{spatialCoords} should be used for all columns. The arguments have
been retained for backward compatibility but may be removed in the future.)}
}
\value{
A \code{SpatialExperiment} object.
}
\description{
The \code{SpatialExperiment} class is designed to represent spatially
resolved transcriptomics (ST) data. It inherits from the
\code{\link{SingleCellExperiment}} class and is used in the same manner. In
addition, the class supports storage of spatial information via
\code{\link{spatialCoords}} and storage of images via \code{\link{imgData}}.
}
\details{
In this class, rows represent genes, and columns represent spots (for
spot-based ST platforms) or cells (for molecule-based ST platforms). As for
\code{\link{SingleCellExperiment}}, \code{counts} and \code{logcounts} can be
stored in the \code{\link{assays}} slot, and row and column metadata in
\code{\link{rowData}} and \code{\link{colData}}. For molecule-based ST data,
the additional measurements per molecule per cell can be stored in a
\code{BumpyMatrix}-formatted \code{assay} named \code{\link{molecules}}.

The additional arguments in the constructor documented above (e.g.
\code{spatialCoords}, \code{imgData}, and others) represent the extensions to
the \code{\link{SingleCellExperiment}} class to store associated spatial and
imaging information for ST data.

The constructor expects \code{colData} to contain a column named
\code{sample_id}. If this is not present, it will assign the value from the
\code{sample_id} argument. If the \code{imgData} argument is provided, the
constructor expects the \code{\link{imgData}} \code{\link{DataFrame}} to
already be built. Otherwise, it will build it from the \code{imageSources}
and (optional) \code{image_id} arguments. If \code{image_id} is not provided,
this will be assumed from \code{sample_id} and \code{imageSources} instead.
To combine multiple samples within a single object, see
\code{\link{combine}}.

For 10x Genomics Visium datasets, the function \code{\link{read10xVisium}}
can be used to load data and create a \code{SpatialExperiment} object
directly from Space Ranger output files.
}
\examples{
#########################################################
# Example 1: Spot-based ST (10x Visium) using constructor
#########################################################

dir <- system.file(
    file.path("extdata", "10xVisium", "section1", "outs"),
    package = "SpatialExperiment")

# read in counts
fnm <- file.path(dir, "raw_feature_bc_matrix")
sce <- DropletUtils::read10xCounts(fnm)

# read in image data
img <- readImgData(
    path = file.path(dir, "spatial"),
    sample_id="foo")

# read in spatial coordinates
fnm <- file.path(dir, "spatial", "tissue_positions_list.csv")
xyz <- read.csv(fnm, header = FALSE,
    col.names = c(
        "barcode", "in_tissue", "array_row", "array_col",
        "pxl_row_in_fullres", "pxl_col_in_fullres"))
    
# construct observation & feature metadata
rd <- S4Vectors::DataFrame(
    symbol = rowData(sce)$Symbol)
  
# construct 'SpatialExperiment'
(spe <- SpatialExperiment(
    assays = list(counts = assay(sce)),
    rowData = rd,
    colData = DataFrame(xyz),
    spatialCoordsNames = c("pxl_col_in_fullres", "pxl_row_in_fullres"),
    imgData = img,
    sample_id = "foo"))
    
#############################################################
# Example 2: Spot-based ST (10x Visium) using 'read10xVisium'
#############################################################

# see ?read10xVisium for details
example(read10xVisium)

##############################
# Example 3: Molecule-based ST
##############################

# create simulated data
n <- 1000; ng <- 50; nc <- 20
# sample xy-coordinates in [0,1]
x <- runif(n)
y <- runif(n)
# assign each molecule to some gene-cell pair
gs <- paste0("gene", seq(ng))
cs <- paste0("cell", seq(nc))
gene <- sample(gs, n, TRUE)
cell <- sample(cs, n, TRUE)
# construct data.frame of molecule coordinates
df <- data.frame(gene, cell, x, y)

# (assure gene & cell are factor so that
# missing observations aren't dropped)
df$gene <- factor(df$gene, gs)
df$cell <- factor(df$cell, cs)

# construct BumpyMatrix
mol <- BumpyMatrix::splitAsBumpyMatrix(
    df[, c("x", "y")],
    row = df$gene, column = df$cell)

# get count matrix
y <- with(df, table(gene, cell))
y <- as.matrix(unclass(y))

# construct SpatialExperiment
(spe_mol <- SpatialExperiment(
    assays = list(
        counts = y, 
        molecules = mol)))
}
\seealso{
\code{?"\link{SpatialExperiment-methods}"}, which includes:
\code{\link{spatialCoords}}, \code{\link{spatialCoordsNames}},
\code{\link{imgData}}, \code{\link{scaleFactors}}

\code{?"\link{SpatialExperiment-assays}"}, which includes:
\code{\link{molecules}}

\code{?"\link{SpatialExperiment-colData}"}

\code{?"\link{SpatialExperiment-combine}"}

\code{?"\link{SpatialExperiment-subset}"}

\code{?"\link{SpatialExperiment-misc}"}

\code{\link{readImgData}}

\code{?"\link{imgData-methods}"}

\code{\link{SpatialImage}}

\code{\link{read10xVisium}}
}
\author{
Dario Righelli and Helena L. Crowell
}
