! This file is part of xtb.
!
! Copyright (C) 2019-2020 Sebastian Ehlert
!
! xtb is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! xtb is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with xtb.  If not, see <https://www.gnu.org/licenses/>.

!> TODO: GFN0-xTB parametrisation data
module xtb_xtb_gfn0
   use xtb_mctc_accuracy, only : wp
   use xtb_param_atomicrad, only : atomicRad
   use xtb_type_param, only : TxTBParameter
   use xtb_xtb_data
   use xtb_xtb_gfn2, only : setGFN2ReferenceOcc
   implicit none
   private

   public :: initGFN0, gfn0Globals
   public :: setGFN0NumberOfPrimitives, setGFN0PairParam


   interface initGFN0
      module procedure :: initData
      module procedure :: initRepulsion
      module procedure :: initCoulomb
      module procedure :: initHamiltonian
   end interface initGFN0

   real(wp), parameter :: enshell(4) = [0.6_wp, -0.1_wp, -0.2_wp, -0.2_wp]
   real(wp), parameter :: kshell(4) = [2.0000000_wp, 2.4868000_wp, 2.2700000_wp, 0.6000000_wp]

   type(TxTBParameter), parameter :: gfn0Globals = TxTBParameter( &
      kshell   = kshell, &
      kdiffa   = 0.0000000_wp, &
      kdiffb   =-0.1000000_wp, &
      enshell  = enshell, &
      enscale4 = 4.0000000_wp, &
      ipeashift= 1.7806900_wp, &
      zcnf     = 0.0537000_wp, &
      tscal    =-0.0129000_wp, &
      kcn      = 3.4847000_wp, &
      fpol     = 0.5097000_wp, &
      ken      = 0.0000000_wp, &
      lshift   = 0.0000000_wp, &
      lshifta  = 0.0000000_wp, &
      split    = 0.0000000_wp, &
      zqf      = 0.0000000_wp, &
      alphaj   = 1.1241000_wp, &
      kexpo    =-0.2000000_wp, &
      dispa    = 0.8000000_wp, &
      dispb    = 4.6000000_wp, &
      dispc    = 2.8500000_wp, &
      dispatm  = 0.0000000_wp, &
      renscale =-0.0900000_wp)

   !> Maximum number of elements supported by GFN0-xTB
   integer, parameter :: maxElem = 86

   ! ========================================================================
   ! REPULSION DATA
   !>
   real(wp), parameter :: kExp = 1.5_wp

   !>
   real(wp), parameter :: kExpLight = kExp

   !>
   real(wp), parameter :: rExp = 1.0_wp

   !>
   real(wp), parameter :: repAlpha(1:maxElem) = [&
      & 2.1885472_wp, 2.2714498_wp, 0.6634645_wp, 0.9267640_wp, 1.1164621_wp, &
      & 1.2680750_wp, 1.6211038_wp, 2.1037547_wp, 2.2062651_wp, 1.9166982_wp, &
      & 0.8129781_wp, 0.8408742_wp, 0.8361156_wp, 0.8859465_wp, 1.0684151_wp, &
      & 1.1882871_wp, 1.4429448_wp, 1.1993811_wp, 0.5700050_wp, 0.8345430_wp, &
      & 0.6840185_wp, 0.7915733_wp, 1.0676223_wp, 0.9216746_wp, 1.1151815_wp, &
      & 1.1883881_wp, 1.1895339_wp, 1.2692713_wp, 1.1734165_wp, 1.0018764_wp, &
      & 1.1597304_wp, 1.1708353_wp, 1.2085038_wp, 1.1161800_wp, 1.3193094_wp, &
      & 0.7670615_wp, 0.6171015_wp, 0.8421909_wp, 0.6513468_wp, 0.6906528_wp, &
      & 0.8705783_wp, 0.9711021_wp, 1.0252504_wp, 0.9847071_wp, 1.0559061_wp, &
      & 1.0645317_wp, 0.9139636_wp, 0.9095541_wp, 0.9965441_wp, 1.0676257_wp, &
      & 1.0759855_wp, 0.8659486_wp, 0.9301733_wp, 0.8139884_wp, 0.5842740_wp, &
      & 0.8070627_wp, 0.6961124_wp, 0.7599095_wp, 0.7667071_wp, 0.7735047_wp, &
      & 0.7803023_wp, 0.7870999_wp, 0.7938975_wp, 0.8006951_wp, 0.8074927_wp, &
      & 0.8142903_wp, 0.8210879_wp, 0.8278855_wp, 0.8346831_wp, 0.8414808_wp, &
      & 0.8482784_wp, 0.8803684_wp, 0.9915500_wp, 0.9875716_wp, 1.1535600_wp, &
      & 1.1418384_wp, 1.1434832_wp, 1.1783705_wp, 1.0591477_wp, 0.9794378_wp, &
      & 1.2439938_wp, 1.0437958_wp, 1.1391049_wp, 0.9115474_wp, 0.9157573_wp, &
      & 0.8137168_wp]

   !>
   real(wp), parameter :: repZeff(1:maxElem) = [&
      & 1.2455414_wp,  1.3440060_wp,  1.1710492_wp,  2.9064151_wp,  4.4020866_wp, &
      & 4.3101011_wp,  4.5460146_wp,  4.7850603_wp,  7.3393960_wp,  4.2503997_wp, &
      &10.5220970_wp,  7.7916659_wp, 11.3886282_wp, 13.9495563_wp, 16.7912135_wp, &
      &13.3874290_wp, 13.9700526_wp, 14.4971987_wp, 13.8061512_wp, 13.9719788_wp, &
      &10.9127447_wp, 13.4067871_wp, 16.7322903_wp, 21.8192969_wp, 22.8754319_wp, &
      &25.2196212_wp, 26.9753662_wp, 27.2652026_wp, 26.2195102_wp, 14.3840374_wp, &
      &25.4102208_wp, 43.7565690_wp, 34.9344472_wp, 22.8724870_wp, 34.2378269_wp, &
      &15.1027639_wp, 39.1086736_wp, 32.7340796_wp, 18.6398784_wp, 22.6163764_wp, &
      &27.6545601_wp, 37.8625561_wp, 40.9844265_wp, 30.0686254_wp, 35.5737255_wp, &
      &28.4443233_wp, 25.9740558_wp, 28.8257081_wp, 53.9657064_wp, 88.0203443_wp, &
      &82.7978295_wp, 39.3120212_wp, 49.7072042_wp, 45.1199137_wp, 55.2536842_wp, &
      &50.0381164_wp, 48.0939804_wp, 46.1827790_wp, 46.0844595_wp, 45.9861400_wp, &
      &45.8878205_wp, 45.7895010_wp, 45.6911815_wp, 45.5928620_wp, 45.4945424_wp, &
      &45.3962229_wp, 45.2979034_wp, 45.1995839_wp, 45.1012644_wp, 45.0029449_wp, &
      &44.9046254_wp, 41.1538255_wp, 46.6524574_wp, 53.4995959_wp, 73.8197012_wp, &
      &59.6567627_wp, 50.0720023_wp, 49.4064531_wp, 44.5201114_wp, 39.7677937_wp, &
      &58.8051943_wp,103.0123579_wp, 85.5566053_wp, 70.6036525_wp, 82.8260761_wp, &
      &68.9676875_wp]

   ! ========================================================================
   ! COULOMB DATA
   !> Electronegativity for the EEQ model
   real(wp), parameter :: eeqChi(1:maxElem) = [&
      & 1.2500000_wp, 1.2912463_wp, 0.8540050_wp, 1.1723939_wp, 1.1094487_wp, &
      & 1.3860275_wp, 1.5341534_wp, 1.5378836_wp, 1.5890750_wp, 1.2893646_wp, &
      & 0.7891208_wp, 0.9983021_wp, 0.9620847_wp, 1.0441134_wp, 1.4789559_wp, &
      & 1.3926377_wp, 1.4749100_wp, 1.2250415_wp, 0.8162292_wp, 1.1252036_wp, &
      & 0.9641451_wp, 0.8810155_wp, 0.9741986_wp, 1.1029038_wp, 1.0076949_wp, &
      & 0.7744353_wp, 0.7554040_wp, 1.0182630_wp, 1.0316167_wp, 1.6317474_wp, &
      & 1.1186739_wp, 1.0345958_wp, 1.3090772_wp, 1.4119283_wp, 1.4500674_wp, &
      & 1.1746889_wp, 0.6686200_wp, 1.0744648_wp, 0.9107813_wp, 0.7876056_wp, &
      & 1.0039889_wp, 0.9225265_wp, 0.9035515_wp, 1.0332301_wp, 1.0293975_wp, &
      & 1.0549549_wp, 1.2356867_wp, 1.2793315_wp, 1.1145650_wp, 1.1214927_wp, &
      & 1.2123167_wp, 1.4003158_wp, 1.4255511_wp, 1.1640198_wp, 0.4685133_wp, &
      & 1.0687873_wp, 0.9335398_wp, 1.0573550_wp, 1.0532043_wp, 1.0490537_wp, &
      & 1.0449031_wp, 1.0407524_wp, 1.0366018_wp, 1.0324512_wp, 1.0283005_wp, &
      & 1.0241499_wp, 1.0199992_wp, 1.0158486_wp, 1.0116980_wp, 1.0075473_wp, &
      & 1.0033967_wp, 0.8612827_wp, 1.0422031_wp, 0.7633168_wp, 0.6019707_wp, &
      & 0.7499393_wp, 0.9511744_wp, 0.9357472_wp, 1.3555382_wp, 1.2006726_wp, &
      & 1.2092025_wp, 1.1736669_wp, 1.1936584_wp, 1.3045488_wp, 1.1964604_wp, &
      & 1.2653792_wp]

   !> Chemical hardness for the EEQ model
   real(wp), parameter :: eeqGam(1:maxElem) = [&
      &-0.3023159_wp, 0.7743046_wp, 0.5303164_wp, 0.2176474_wp, 0.1956176_wp, &
      & 0.0308461_wp, 0.0559522_wp, 0.0581228_wp, 0.1574017_wp, 0.6825784_wp, &
      & 0.3922376_wp, 0.5581866_wp, 0.3017510_wp, 0.1039137_wp, 0.2124917_wp, &
      & 0.0580720_wp, 0.2537467_wp, 0.5780354_wp, 0.3920658_wp,-0.0024897_wp, &
      &-0.0061520_wp, 0.1663252_wp, 0.1051751_wp, 0.0009900_wp, 0.0976543_wp, &
      & 0.0612028_wp, 0.0561526_wp, 0.0899774_wp, 0.1313171_wp, 0.5728071_wp, &
      & 0.1741615_wp, 0.2671888_wp, 0.2351989_wp, 0.0718104_wp, 0.3458143_wp, &
      & 0.8203265_wp, 0.4287770_wp, 0.2667067_wp, 0.0873658_wp, 0.0599431_wp, &
      & 0.1581972_wp, 0.1716374_wp, 0.2721649_wp, 0.2817608_wp, 0.1391572_wp, &
      & 0.1175925_wp, 0.2316104_wp, 0.2256303_wp, 0.1230459_wp, 0.0141941_wp, &
      & 0.0188612_wp, 0.0230207_wp, 0.3644113_wp, 0.1668461_wp, 0.5167533_wp, &
      & 0.1979578_wp, 0.0345176_wp, 0.0240233_wp, 0.0246333_wp, 0.0252433_wp, &
      & 0.0258532_wp, 0.0264632_wp, 0.0270732_wp, 0.0276832_wp, 0.0282931_wp, &
      & 0.0289031_wp, 0.0295131_wp, 0.0301230_wp, 0.0307330_wp, 0.0313430_wp, &
      & 0.0319529_wp, 0.0262881_wp, 0.1715396_wp, 0.1803633_wp, 0.3631824_wp, &
      & 0.3010980_wp, 0.1100299_wp, 0.0277514_wp, 0.0554975_wp, 0.7723231_wp, &
      & 0.1287718_wp, 0.1034598_wp, 0.0114935_wp, 0.0160842_wp, 0.3369611_wp, &
      & 0.1844179_wp]

   !> Coordination number dependence of the EN in the EEQ model
   real(wp), parameter :: eeqkCN(1:maxElem) = [&
      & 0.0248762_wp, 0.1342276_wp, 0.0103048_wp,-0.0352374_wp,-0.0980031_wp, &
      & 0.0643920_wp, 0.1053273_wp, 0.1394809_wp, 0.1276675_wp,-0.1081936_wp, &
      &-0.0008132_wp,-0.0279860_wp,-0.0521436_wp,-0.0257206_wp, 0.1651461_wp, &
      & 0.0914418_wp, 0.1213634_wp,-0.0636298_wp,-0.0045838_wp, 0.0007509_wp, &
      &-0.0307730_wp,-0.0286150_wp,-0.0341465_wp,-0.0419655_wp,-0.0088536_wp, &
      &-0.1001069_wp,-0.1190502_wp,-0.0726233_wp,-0.0219233_wp, 0.0641913_wp, &
      &-0.0103130_wp, 0.0262628_wp, 0.0222202_wp, 0.0709954_wp, 0.0422244_wp, &
      &-0.0308245_wp, 0.0086249_wp,-0.0237146_wp,-0.0721798_wp,-0.0848810_wp, &
      &-0.0402828_wp,-0.0372396_wp,-0.0027043_wp, 0.0525839_wp, 0.0051192_wp, &
      & 0.0188401_wp, 0.0103998_wp, 0.0000549_wp, 0.0087717_wp,-0.0237228_wp, &
      & 0.0169656_wp, 0.0924186_wp, 0.0352884_wp,-0.0091444_wp, 0.0192916_wp, &
      &-0.0154483_wp,-0.0736833_wp,-0.0064191_wp,-0.0093012_wp,-0.0121833_wp, &
      &-0.0150654_wp,-0.0179475_wp,-0.0208296_wp,-0.0237117_wp,-0.0265938_wp, &
      &-0.0294759_wp,-0.0323580_wp,-0.0352400_wp,-0.0381221_wp,-0.0410042_wp, &
      &-0.0438863_wp,-0.0894776_wp,-0.0333583_wp,-0.0154963_wp,-0.0121092_wp, &
      &-0.0744239_wp, 0.0050138_wp,-0.0153757_wp,-0.0029221_wp, 0.0239125_wp, &
      & 0.0183012_wp,-0.0238011_wp,-0.0268025_wp, 0.0136505_wp,-0.0132199_wp, &
      &-0.0439890_wp]

   !> Charge width in the EEQ model
   real(wp), parameter :: eeqAlp(1:maxElem) = [&
      & 0.7490227_wp, 0.4196569_wp, 1.4256190_wp, 2.0698743_wp, 1.7358798_wp, &
      & 1.8288757_wp, 1.9346081_wp, 1.6974795_wp, 0.8169179_wp, 0.6138441_wp, &
      & 1.7294046_wp, 1.7925036_wp, 1.2156739_wp, 1.5314457_wp, 1.3730859_wp, &
      & 1.7936326_wp, 2.4255996_wp, 1.5891656_wp, 2.1829647_wp, 1.4177623_wp, &
      & 1.5181399_wp, 1.9919805_wp, 1.7171675_wp, 2.0655063_wp, 1.3318009_wp, &
      & 1.3660068_wp, 1.5694128_wp, 1.2762644_wp, 1.0039549_wp, 0.7338863_wp, &
      & 3.2596250_wp, 1.7530299_wp, 1.5281792_wp, 2.1837813_wp, 2.1642027_wp, &
      & 2.7280594_wp, 0.7838049_wp, 1.4274742_wp, 1.8023947_wp, 1.6093288_wp, &
      & 1.3834349_wp, 1.1740977_wp, 1.5768259_wp, 1.3205263_wp, 1.4259466_wp, &
      & 1.1499748_wp, 0.7013009_wp, 1.2374416_wp, 1.3799991_wp, 1.8528424_wp, &
      & 1.8497568_wp, 2.0159294_wp, 1.2903708_wp, 2.0199161_wp, 0.9530522_wp, &
      & 1.5015025_wp, 2.1917012_wp, 1.9134370_wp, 1.9897910_wp, 2.0661450_wp, &
      & 2.1424991_wp, 2.2188531_wp, 2.2952071_wp, 2.3715611_wp, 2.4479151_wp, &
      & 2.5242691_wp, 2.6006231_wp, 2.6769771_wp, 2.7533312_wp, 2.8296852_wp, &
      & 2.9060392_wp, 1.6423047_wp, 1.3567622_wp, 1.8966648_wp, 0.8253100_wp, &
      & 0.7412219_wp, 1.0350883_wp, 0.9692278_wp, 1.0048087_wp, 2.3138674_wp, &
      & 2.8055966_wp, 3.0968677_wp, 1.6597596_wp, 3.2191575_wp, 1.5388148_wp, &
      & 2.1222013_wp]

   ! ========================================================================
   ! HAMILTONIAN DATA
   !> Number of shells
   integer, parameter :: nShell(1:maxElem) = [&
      & 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 3, 3, 3, 3, 3, 3, 3, 2, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, &
      & 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, &
      & 2, 2, 2, 2, 3, 3]

   !> Angular momentum of each shell
   integer, parameter :: angShell(3, 1:maxElem) = reshape([&
      & 0, 0, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  &
      & 0, 1, 0,  0, 1, 0,  0, 1, 2,  0, 1, 0,  0, 1, 2,  0, 1, 2,  0, 1, 2,  &
      & 0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 0,  0, 1, 2,  2, 0, 1,  &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  &
      & 2, 0, 1,  0, 1, 0,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  &
      & 0, 1, 2,  0, 1, 0,  0, 1, 2,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  0, 1, 0,  0, 1, 2,  &
      & 0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 2,  0, 1, 0,  0, 1, 2,  &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  &
      & 2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  2, 0, 1,  &
      & 2, 0, 1,  2, 0, 1,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  0, 1, 0,  &
      & 0, 1, 2,  0, 1, 2], shape(angShell))

   !> Principal quantum number of each shell
   integer, parameter :: principalQuantumNumber(3, 1:maxElem) = reshape([&
      & 1, 2, 0,  1, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0,  2, 2, 0,  &
      & 2, 2, 0,  2, 2, 0,  2, 2, 3,  3, 3, 0,  3, 3, 3,  3, 3, 3,  3, 3, 3,  &
      & 3, 3, 3,  3, 3, 3,  3, 3, 3,  3, 3, 3,  4, 4, 0,  4, 4, 3,  3, 4, 4,  &
      & 3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  3, 4, 4,  &
      & 3, 4, 4,  4, 4, 0,  4, 4, 4,  4, 4, 4,  4, 4, 4,  4, 4, 4,  4, 4, 4,  &
      & 4, 4, 4,  5, 5, 0,  5, 5, 4,  4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5,  &
      & 4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5,  4, 5, 5,  5, 5, 0,  5, 5, 5,  &
      & 5, 5, 5,  5, 5, 5,  5, 5, 5,  5, 5, 5,  5, 5, 5,  6, 6, 0,  6, 6, 5,  &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  &
      & 5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  5, 6, 6,  &
      & 5, 6, 6,  5, 6, 6,  6, 6, 0,  6, 6, 0,  6, 6, 0,  6, 6, 0,  6, 6, 0,  &
      & 6, 6, 5,  6, 6, 5], shape(principalQuantumNumber))

   !>
   real(wp), parameter :: electronegativity(1:maxElem) = [&
      & 1.92_wp,  3.00_wp,  0.98_wp,  1.57_wp,  2.04_wp, &
      & 2.48_wp,  2.97_wp,  3.44_wp,  3.50_wp,  3.50_wp, &
      & 0.93_wp,  1.31_wp,  1.61_wp,  1.90_wp,  2.19_wp, &
      & 2.58_wp,  3.16_wp,  3.50_wp,  1.45_wp,  1.80_wp, &
      & 1.73_wp,  1.54_wp,  1.63_wp,  1.66_wp,  1.55_wp, &
      & 1.83_wp,  1.88_wp,  1.91_wp,  1.90_wp,  1.65_wp, &
      & 1.81_wp,  2.01_wp,  2.18_wp,  2.55_wp,  2.96_wp, &
      & 3.00_wp,  1.50_wp,  1.50_wp,  1.55_wp,  1.33_wp, &
      & 1.60_wp,  2.16_wp,  1.90_wp,  2.20_wp,  2.28_wp, &
      & 2.20_wp,  1.93_wp,  1.69_wp,  1.78_wp,  1.96_wp, &
      & 2.05_wp,  2.10_wp,  2.66_wp,  2.60_wp,  1.50_wp, &
      & 1.60_wp,  1.50_wp,  1.50_wp,  1.50_wp,  1.50_wp, &
      & 1.50_wp,  1.50_wp,  1.50_wp,  1.50_wp,  1.50_wp, &
      & 1.50_wp,  1.50_wp,  1.50_wp,  1.50_wp,  1.50_wp, &
      & 1.50_wp,  1.30_wp,  1.50_wp,  2.36_wp,  1.90_wp, &
      & 2.20_wp,  2.20_wp,  2.28_wp,  2.54_wp,  2.00_wp, &
      & 1.62_wp,  2.33_wp,  2.02_wp,  2.00_wp,  2.20_wp, &
      & 2.20_wp]
 
   real(wp), parameter :: shellPoly(0:3, 1:maxElem) = reshape([&
      & -3.6355142_wp,  0.00000000_wp,  0.0000000_wp,  0.0000000_wp, &
      &-12.4694067_wp,  5.12004200_wp,  0.0000000_wp,  0.0000000_wp, &
      & -9.1309270_wp, 26.45499410_wp,  0.0000000_wp,  0.0000000_wp, &
      &-23.0556019_wp, 12.29400490_wp,  0.0000000_wp,  0.0000000_wp, &
      &-13.0296301_wp, -3.80008460_wp,  0.0000000_wp,  0.0000000_wp, &
      &-12.8096830_wp, -8.82326510_wp,  0.0000000_wp,  0.0000000_wp, &
      &-12.6808155_wp, -8.80380760_wp,  0.0000000_wp,  0.0000000_wp, &
      &-16.1325184_wp, -5.51085690_wp,  0.0000000_wp,  0.0000000_wp, &
      & -1.9980582_wp,-15.34156120_wp,  0.0000000_wp,  0.0000000_wp, &
      &-19.0634182_wp,-17.94052850_wp,  2.9172938_wp,  0.0000000_wp, &
      & -1.9038223_wp, 20.51582340_wp,  0.0000000_wp,  0.0000000_wp, &
      &-23.8821588_wp,  6.21983350_wp, 16.9907381_wp,  0.0000000_wp, &
      &-25.9314993_wp, 10.02183490_wp,  5.7458163_wp,  0.0000000_wp, &
      &-14.0752116_wp,-18.03006400_wp, 13.1379417_wp,  0.0000000_wp, &
      &-22.5131801_wp,-15.13429710_wp, 31.3131980_wp,  0.0000000_wp, &
      &-27.9640046_wp,-12.62869290_wp, 25.8298106_wp,  0.0000000_wp, &
      &-25.6075631_wp,-14.26984150_wp, 37.2632140_wp,  0.0000000_wp, &
      &-26.8745198_wp,-16.37147400_wp, 21.6207312_wp,  0.0000000_wp, &
      &  1.7106901_wp, 24.43233360_wp,  0.0000000_wp,  0.0000000_wp, &
      &-21.2288327_wp, -3.23987870_wp, 13.9882752_wp,  0.0000000_wp, &
      &  5.3449585_wp, 25.37163520_wp,-31.0302574_wp,  0.0000000_wp, &
      & 21.1807596_wp, 24.93452780_wp,-26.2283413_wp,  0.0000000_wp, &
      &  7.0558197_wp, 26.90006100_wp,-31.3799774_wp,  0.0000000_wp, &
      &  2.0452996_wp, 27.02537350_wp,-15.1553980_wp,  0.0000000_wp, &
      & -2.9359221_wp, 29.37855170_wp,-27.4411979_wp,  0.0000000_wp, &
      &  6.2785647_wp, 28.40737540_wp,-25.0617776_wp,  0.0000000_wp, &
      &  2.7815688_wp, 22.85568860_wp,-22.1545572_wp,  0.0000000_wp, &
      & -6.3873757_wp, 18.50679430_wp,-20.6965198_wp,  0.0000000_wp, &
      & -9.3513228_wp,  8.03612800_wp,-25.4274699_wp,  0.0000000_wp, &
      &-20.5283728_wp, 18.31637340_wp,  0.0000000_wp,  0.0000000_wp, &
      &-25.9659007_wp,  0.24932040_wp, 31.9811359_wp,  0.0000000_wp, &
      &-22.5969796_wp,-13.07779110_wp, 29.8243400_wp,  0.0000000_wp, &
      &-27.5975320_wp,-20.15791270_wp, 33.9750529_wp,  0.0000000_wp, &
      &-24.4009201_wp,-18.49323350_wp, 31.6569329_wp,  0.0000000_wp, &
      &-22.8480620_wp,-20.46239210_wp, 32.3160404_wp,  0.0000000_wp, &
      &-29.6544708_wp,-11.81015280_wp, 26.3649672_wp,  0.0000000_wp, &
      & -3.9550260_wp, 17.81029220_wp,  0.0000000_wp,  0.0000000_wp, &
      &-20.4705547_wp, -5.66237750_wp,  1.1533441_wp,  0.0000000_wp, &
      &-14.8904419_wp, 32.75122970_wp,-24.8248233_wp,  0.0000000_wp, &
      & -1.2323670_wp, 31.35218880_wp,-25.4108005_wp,  0.0000000_wp, &
      &-20.1589551_wp, 23.91255900_wp,-28.9368791_wp,  0.0000000_wp, &
      & 10.4647630_wp, 25.70713940_wp,-27.1881494_wp,  0.0000000_wp, &
      & 17.2824035_wp, 26.97508400_wp,-27.0119119_wp,  0.0000000_wp, &
      &  6.3119382_wp, 27.76646610_wp,-26.3813902_wp,  0.0000000_wp, &
      &  1.1913117_wp, 25.13980880_wp,-19.0998367_wp,  0.0000000_wp, &
      &  7.5528381_wp, 31.60282550_wp,-24.7810390_wp,  0.0000000_wp, &
      & -7.4614740_wp, 11.91715850_wp,-24.0147360_wp,  0.0000000_wp, &
      &-21.1131335_wp, 24.22912690_wp,  0.0000000_wp,  0.0000000_wp, &
      &-28.0671409_wp, -2.16073050_wp, 30.6807055_wp,  0.0000000_wp, &
      &-21.2192117_wp, -7.23565880_wp, 12.8333865_wp,  0.0000000_wp, &
      &-25.9808690_wp,-16.80154040_wp, 34.5734795_wp,  0.0000000_wp, &
      &-21.3500803_wp,-20.83440960_wp, 36.7357631_wp,  0.0000000_wp, &
      &-27.3671183_wp,-19.62221140_wp, 33.1425727_wp,  0.0000000_wp, &
      &-29.8202170_wp,-16.61175030_wp, 21.1953313_wp,  0.0000000_wp, &
      &-12.1334296_wp, 18.86964700_wp,  0.0000000_wp,  0.0000000_wp, &
      &-22.8455123_wp, -9.98368600_wp,  3.4483973_wp,  0.0000000_wp, &
      &-15.6183059_wp, 33.47909370_wp,-25.5526873_wp,  0.0000000_wp, &
      &-15.6451743_wp, 25.92955660_wp,-31.4845028_wp,  0.0000000_wp, &
      &-15.6344636_wp, 26.14183290_wp,-31.4071167_wp,  0.0000000_wp, &
      &-15.6237528_wp, 26.35410920_wp,-31.3297306_wp,  0.0000000_wp, &
      &-15.6130421_wp, 26.56638560_wp,-31.2523445_wp,  0.0000000_wp, &
      &-15.6023314_wp, 26.77866190_wp,-31.1749584_wp,  0.0000000_wp, &
      &-15.5916207_wp, 26.99093820_wp,-31.0975723_wp,  0.0000000_wp, &
      &-15.5809100_wp, 27.20321460_wp,-31.0201862_wp,  0.0000000_wp, &
      &-15.5701992_wp, 27.41549090_wp,-30.9428001_wp,  0.0000000_wp, &
      &-15.5594885_wp, 27.62776720_wp,-30.8654140_wp,  0.0000000_wp, &
      &-15.5487778_wp, 27.84004360_wp,-30.7880279_wp,  0.0000000_wp, &
      &-15.5380671_wp, 28.05231990_wp,-30.7106418_wp,  0.0000000_wp, &
      &-15.5273563_wp, 28.26459620_wp,-30.6332557_wp,  0.0000000_wp, &
      &-15.5166456_wp, 28.47687260_wp,-30.5558696_wp,  0.0000000_wp, &
      &-15.5059349_wp, 28.68914890_wp,-30.4784835_wp,  0.0000000_wp, &
      & -3.9147090_wp, 32.69618100_wp,-32.9129200_wp,  0.0000000_wp, &
      &-22.3008469_wp, 29.83540160_wp,-29.6033096_wp,  0.0000000_wp, &
      &  0.9644129_wp, 45.94330280_wp,-24.8055364_wp,  0.0000000_wp, &
      & 30.3500314_wp, 38.21103860_wp,-28.0450462_wp,  0.0000000_wp, &
      &  7.6941848_wp, 24.23188770_wp,-27.3810511_wp,  0.0000000_wp, &
      &  2.4984632_wp, 17.48460760_wp,-24.0537099_wp,  0.0000000_wp, &
      &  3.2909380_wp, 15.99839670_wp,-23.9093136_wp,  0.0000000_wp, &
      &-10.1996894_wp, 15.40677260_wp,-18.2423967_wp,  0.0000000_wp, &
      &-20.1460848_wp, 21.07209230_wp,  0.0000000_wp,  0.0000000_wp, &
      &-24.8474969_wp,  5.50304010_wp,  0.0000000_wp,  0.0000000_wp, &
      &-27.4676742_wp, -6.24492100_wp,  0.0000000_wp,  0.0000000_wp, &
      &-27.3029972_wp,-13.99614710_wp,  0.0000000_wp,  0.0000000_wp, &
      &-19.9670108_wp,-18.36130340_wp,  0.0000000_wp,  0.0000000_wp, &
      &-32.1472952_wp,-20.79729510_wp, 27.0343912_wp,  0.0000000_wp, &
      &-33.4702358_wp,-18.44551930_wp, 22.5007703_wp,  0.0000000_wp],&
      & shape(shellPoly))

   real(wp), parameter :: kCN(0:2, 1:maxElem) = reshape([&
      & 0.07116904_wp, 0.00000000_wp, 0.00000000_wp, &
      &-0.21046272_wp,-0.37377613_wp, 0.00000000_wp, &
      &-0.00588643_wp, 0.01229940_wp, 0.00000000_wp, &
      & 0.03659056_wp,-0.00543602_wp, 0.00000000_wp, &
      &-0.30472261_wp, 0.00556489_wp, 0.00000000_wp, &
      &-0.55477603_wp, 0.15631408_wp, 0.00000000_wp, &
      &-0.02818725_wp, 0.06027542_wp, 0.00000000_wp, &
      &-0.12537900_wp,-0.02934837_wp, 0.00000000_wp, &
      & 0.11952872_wp,-0.02001469_wp, 0.00000000_wp, &
      &-0.07969349_wp,-0.03183102_wp,-0.02598776_wp, &
      & 0.06047597_wp, 0.01676901_wp, 0.00000000_wp, &
      & 0.03066908_wp,-0.00315587_wp, 0.03687894_wp, &
      &-0.07913111_wp, 0.03614827_wp,-0.08703979_wp, &
      & 0.18789681_wp,-0.08229514_wp,-0.03042044_wp, &
      & 0.07048836_wp,-0.02076245_wp,-0.09160855_wp, &
      &-0.01240442_wp,-0.00611739_wp,-0.00694984_wp, &
      & 0.11846012_wp,-0.00422548_wp,-0.10808918_wp, &
      &-0.08330348_wp, 0.01406891_wp,-0.02103496_wp, &
      & 0.00431781_wp, 0.02249671_wp, 0.00000000_wp, &
      & 0.09861689_wp, 0.08916793_wp, 0.03360490_wp, &
      &-0.04244589_wp,-0.01039491_wp, 0.11290272_wp, &
      &-0.26131513_wp,-0.09668192_wp, 0.17000000_wp, &
      &-0.03811876_wp,-0.03240250_wp, 0.02890820_wp, &
      &-0.48366149_wp,-0.00547793_wp, 0.00727051_wp, &
      &-0.15528487_wp,-0.00750273_wp, 0.01000000_wp, &
      &-0.30453727_wp,-0.06034576_wp, 0.07000000_wp, &
      &-0.08636611_wp, 0.05404216_wp, 0.10000000_wp, &
      & 0.17130286_wp,-0.08594284_wp, 0.02045707_wp, &
      & 0.02336012_wp,-0.03151736_wp, 0.01150023_wp, &
      & 0.04980615_wp, 0.05813466_wp, 0.00000000_wp, &
      & 0.05567111_wp, 0.05748990_wp,-0.02131931_wp, &
      & 0.18775492_wp,-0.11667497_wp,-0.01468127_wp, &
      & 0.03228237_wp,-0.06260172_wp, 0.01759185_wp, &
      &-0.00314322_wp,-0.04075826_wp, 0.06444935_wp, &
      &-0.03614742_wp, 0.03742490_wp,-0.06477093_wp, &
      &-0.03657585_wp,-0.00288971_wp, 0.00662612_wp, &
      & 0.02927284_wp, 0.04327208_wp, 0.00000000_wp, &
      & 0.08783924_wp, 0.05717898_wp, 0.03126696_wp, &
      & 0.08433168_wp,-0.00730286_wp, 0.03976138_wp, &
      & 0.10938012_wp,-0.06167863_wp, 0.01437786_wp, &
      & 0.15858874_wp,-0.02495775_wp,-0.07270306_wp, &
      & 0.01240865_wp,-0.06514497_wp, 0.08912258_wp, &
      &-0.00695215_wp,-0.02651591_wp, 0.00906910_wp, &
      & 0.03318922_wp,-0.02027599_wp, 0.01037674_wp, &
      & 0.00444124_wp,-0.02052962_wp, 0.01091464_wp, &
      &-0.06534170_wp, 0.00773043_wp, 0.01035578_wp, &
      &-0.02041670_wp,-0.04941126_wp, 0.00507145_wp, &
      & 0.10104763_wp,-0.10742210_wp, 0.00000000_wp, &
      & 0.04647355_wp,-0.00995082_wp,-0.02852205_wp, &
      & 0.16446031_wp,-0.10004245_wp,-0.08549641_wp, &
      &-0.05778929_wp, 0.07863872_wp,-0.00005389_wp, &
      & 0.12681267_wp,-0.04141598_wp, 0.03085079_wp, &
      &-0.07187128_wp, 0.04073371_wp,-0.03576737_wp, &
      &-0.01384278_wp, 0.00156952_wp,-0.00051988_wp, &
      & 0.01307258_wp, 0.00901257_wp, 0.00000000_wp, &
      & 0.05149152_wp, 0.04850276_wp, 0.02866773_wp, &
      & 0.06268874_wp,-0.00242200_wp, 0.02955699_wp, &
      & 0.07756608_wp,-0.00004407_wp, 0.06487575_wp, &
      & 0.07945053_wp,-0.00004327_wp, 0.06284641_wp, &
      & 0.08133498_wp,-0.00004248_wp, 0.06081707_wp, &
      & 0.08321944_wp,-0.00004169_wp, 0.05878773_wp, &
      & 0.08510389_wp,-0.00004089_wp, 0.05675839_wp, &
      & 0.08698834_wp,-0.00004010_wp, 0.05472904_wp, &
      & 0.08887279_wp,-0.00003930_wp, 0.05269970_wp, &
      & 0.09075725_wp,-0.00003851_wp, 0.05067036_wp, &
      & 0.09264170_wp,-0.00003772_wp, 0.04864102_wp, &
      & 0.09452615_wp,-0.00003692_wp, 0.04661168_wp, &
      & 0.09641060_wp,-0.00003613_wp, 0.04458234_wp, &
      & 0.09829505_wp,-0.00003534_wp, 0.04255300_wp, &
      & 0.10017951_wp,-0.00003454_wp, 0.04052365_wp, &
      & 0.10206396_wp,-0.00003375_wp, 0.03849431_wp, &
      & 0.14365678_wp,-0.01607540_wp, 0.02182218_wp, &
      & 0.10205952_wp,-0.02748425_wp,-0.03320988_wp, &
      & 0.29046502_wp, 0.00361699_wp,-0.01600400_wp, &
      & 0.11515463_wp,-0.00359938_wp,-0.03512912_wp, &
      & 0.03746916_wp,-0.00800203_wp,-0.01295399_wp, &
      &-0.01285049_wp, 0.01466161_wp, 0.00829261_wp, &
      & 0.08632843_wp, 0.13889222_wp,-0.01831911_wp, &
      &-0.03352364_wp,-0.06229470_wp, 0.00937469_wp, &
      & 0.06338123_wp,-0.16974940_wp, 0.00000000_wp, &
      &-0.01035095_wp, 0.02609025_wp, 0.00000000_wp, &
      &-0.19929968_wp, 0.17808807_wp, 0.00000000_wp, &
      &-0.02029156_wp, 0.04586688_wp, 0.00000000_wp, &
      &-0.02278953_wp, 0.05476139_wp, 0.00000000_wp, &
      &-0.02597842_wp, 0.00175268_wp, 0.18437642_wp, &
      &-0.00041023_wp, 0.00074457_wp,-0.03563225_wp],&
      & shape(kCN))

   real(wp), parameter :: selfEnergy(3, 1:maxElem) = reshape([&
      &-11.9223639_wp, -2.8061095_wp,  0.0000000_wp, &
      &-20.9532631_wp, -1.1297785_wp,  0.0000000_wp, &
      & -6.9971696_wp, -3.2655741_wp,  0.0000000_wp, &
      & -9.8114460_wp, -4.1691241_wp,  0.0000000_wp, &
      &-11.5274724_wp, -7.1785553_wp,  0.0000000_wp, &
      &-15.7545853_wp, -9.7975356_wp,  0.0000000_wp, &
      &-18.8417536_wp,-11.5390772_wp,  0.0000000_wp, &
      &-17.9332316_wp,-11.8400363_wp,  0.0000000_wp, &
      &-21.1766889_wp,-12.0966829_wp,  0.0000000_wp, &
      &-23.8117049_wp,-12.7315278_wp, -2.6152816_wp, &
      & -8.0168587_wp, -3.5357541_wp,  0.0000000_wp, &
      & -8.8996217_wp, -3.3930267_wp, -2.3704000_wp, &
      &-11.4217432_wp, -5.5010511_wp, -3.7796816_wp, &
      &-14.1305658_wp, -8.2750304_wp, -3.1140941_wp, &
      &-15.7071105_wp, -9.8725694_wp, -1.6270802_wp, &
      &-20.1581637_wp,-11.1864809_wp, -0.8356505_wp, &
      &-26.2723861_wp,-12.3657994_wp, -1.9017158_wp, &
      &-22.0270820_wp,-14.3127631_wp, -2.9389433_wp, &
      & -6.6921296_wp, -3.1081605_wp,  0.0000000_wp, &
      & -8.0531971_wp, -2.1761231_wp, -0.2675983_wp, &
      & -8.7129730_wp, -9.0228950_wp, -2.3954879_wp, &
      & -8.5703364_wp, -9.4933734_wp, -2.2959946_wp, &
      & -8.7570928_wp, -9.8674899_wp, -1.2850031_wp, &
      & -8.8203392_wp, -7.1020286_wp, -1.3383282_wp, &
      & -9.5757701_wp, -6.0802192_wp, -3.7852117_wp, &
      &-10.1463332_wp, -5.5418598_wp, -3.7951519_wp, &
      &-10.5325577_wp, -4.9644527_wp, -2.6210546_wp, &
      &-10.5903637_wp, -6.6362657_wp, -2.3476564_wp, &
      &-11.3598265_wp, -8.4558686_wp, -3.0589423_wp, &
      &-11.0505242_wp, -2.7759459_wp,  0.0000000_wp, &
      &-11.2255436_wp, -4.6379660_wp, -0.5568933_wp, &
      &-15.5640015_wp, -9.1779578_wp, -1.1917480_wp, &
      &-16.8021037_wp,-10.2020806_wp, -0.1526587_wp, &
      &-20.6883851_wp,-11.3475165_wp, -0.0925094_wp, &
      &-19.8975374_wp,-11.6316818_wp, -1.1601961_wp, &
      &-17.7445664_wp,-13.3156057_wp, -1.3999865_wp, &
      & -6.6590644_wp, -3.2967564_wp,  0.0000000_wp, &
      & -6.3552773_wp, -1.6881756_wp, -0.1094950_wp, &
      & -7.3282377_wp,-10.5232537_wp, -1.2841045_wp, &
      & -8.3528075_wp, -9.4151949_wp, -1.9905675_wp, &
      & -8.9909092_wp, -9.3825305_wp, -1.0846865_wp, &
      & -8.3391136_wp, -5.0423265_wp, -3.3157522_wp, &
      & -9.5934551_wp, -4.1238201_wp, -2.8210863_wp, &
      &-10.4178466_wp, -4.6661993_wp, -4.3910247_wp, &
      &-11.2407436_wp, -6.3230998_wp, -2.7898020_wp, &
      &-11.0477513_wp, -7.5245955_wp, -1.3458846_wp, &
      &-10.7029327_wp, -9.0008211_wp, -3.1175174_wp, &
      &-12.0082253_wp, -2.6607618_wp,  0.0000000_wp, &
      &-12.8089312_wp, -4.9517162_wp, -0.6628616_wp, &
      &-20.5912595_wp, -8.8280995_wp, -2.6074284_wp, &
      &-19.1309518_wp, -9.2461857_wp,  0.3120269_wp, &
      &-18.8494281_wp,-11.4037354_wp, -0.5155777_wp, &
      &-17.9130306_wp,-11.6266388_wp, -1.4119025_wp, &
      &-15.8963959_wp,-12.6153098_wp, -0.9187936_wp, &
      & -6.2935338_wp, -3.2602182_wp,  0.0000000_wp, &
      & -5.6796095_wp, -2.5183292_wp, -0.8391169_wp, &
      & -8.1422087_wp,-11.6440990_wp, -0.8239123_wp, &
      & -8.2709324_wp,-11.3009295_wp, -3.2425383_wp, &
      & -8.3037702_wp,-11.1705313_wp, -3.1342334_wp, &
      & -8.3366080_wp,-11.0401330_wp, -3.0259285_wp, &
      & -8.3694458_wp,-10.9097347_wp, -2.9176236_wp, &
      & -8.4022836_wp,-10.7793365_wp, -2.8093186_wp, &
      & -8.4351215_wp,-10.6489382_wp, -2.7010137_wp, &
      & -8.4679593_wp,-10.5185399_wp, -2.5927088_wp, &
      & -8.5007971_wp,-10.3881417_wp, -2.4844039_wp, &
      & -8.5336349_wp,-10.2577434_wp, -2.3760990_wp, &
      & -8.5664727_wp,-10.1273451_wp, -2.2677941_wp, &
      & -8.5993105_wp, -9.9969469_wp, -2.1594891_wp, &
      & -8.6321483_wp, -9.8665486_wp, -2.0511842_wp, &
      & -8.6649861_wp, -9.7361503_wp, -1.9428793_wp, &
      & -8.6978239_wp, -9.6057521_wp, -1.8345744_wp, &
      & -8.3270480_wp, -9.2677188_wp, -0.9691591_wp, &
      & -9.1503016_wp,-10.5247435_wp, -0.9450823_wp, &
      & -9.6425716_wp, -8.4038882_wp, -0.4830525_wp, &
      &-10.2383908_wp, -4.9432345_wp, -1.3549000_wp, &
      &-10.0139372_wp, -5.4818074_wp, -2.6947002_wp, &
      &-11.1406750_wp, -7.5775954_wp, -3.8917827_wp, &
      &-11.3162420_wp, -8.8896294_wp, -3.0510272_wp, &
      &-12.1009764_wp, -9.5056791_wp, -2.2886714_wp, &
      &-12.1695315_wp, -2.6656574_wp,  0.0000000_wp, &
      &-20.1640068_wp, -4.9875823_wp,  0.0000000_wp, &
      &-22.0691007_wp, -8.1233437_wp,  0.0000000_wp, &
      &-19.8457402_wp, -8.1841396_wp,  0.0000000_wp, &
      &-22.7267689_wp,-10.6638874_wp,  0.0000000_wp, &
      &-16.2231111_wp,-10.5823143_wp, -0.4133880_wp, &
      &-13.6425470_wp,-12.1716393_wp, -0.6534981_wp],&
      & shape(selfEnergy))

   real(wp), parameter :: slaterExponent(3, 1:maxElem) = reshape([&
      & 1.2209169_wp, 2.0165983_wp, 0.0000000_wp, &
      & 1.5824890_wp, 1.6000000_wp, 0.0000000_wp, &
      & 0.5005719_wp, 0.4658228_wp, 0.0000000_wp, &
      & 0.9078603_wp, 1.0039670_wp, 0.0000000_wp, &
      & 1.7660778_wp, 1.6244411_wp, 0.0000000_wp, &
      & 1.9915841_wp, 1.7845353_wp, 0.0000000_wp, &
      & 2.3225221_wp, 2.0626548_wp, 0.0000000_wp, &
      & 2.6718248_wp, 2.2239915_wp, 0.0000000_wp, &
      & 2.6161081_wp, 2.4319703_wp, 0.0000000_wp, &
      & 2.9809477_wp, 2.0785656_wp, 1.8460393_wp, &
      & 0.5467934_wp, 0.5633952_wp, 0.0000000_wp, &
      & 1.0300807_wp, 0.7916346_wp, 0.8488197_wp, &
      & 1.2874433_wp, 1.0904909_wp, 0.8900769_wp, &
      & 1.4531665_wp, 1.9693476_wp, 1.0819287_wp, &
      & 1.8253512_wp, 1.7857588_wp, 1.6661366_wp, &
      & 1.8928587_wp, 1.9498486_wp, 1.6934107_wp, &
      & 2.4427851_wp, 2.0467885_wp, 2.2975192_wp, &
      & 2.4411522_wp, 2.0047267_wp, 1.9691892_wp, &
      & 0.7780770_wp, 0.7000000_wp, 0.0000000_wp, &
      & 1.1200000_wp, 0.7483882_wp, 1.5500000_wp, &
      & 1.6075504_wp, 1.0933429_wp, 1.0700000_wp, &
      & 1.5158150_wp, 1.2148492_wp, 1.0965783_wp, &
      & 1.7421878_wp, 1.1888016_wp, 0.8934693_wp, &
      & 1.8634588_wp, 1.4228855_wp, 1.4932821_wp, &
      & 1.7327124_wp, 1.4062701_wp, 1.2782309_wp, &
      & 1.9253779_wp, 0.9262150_wp, 1.2346615_wp, &
      & 2.1305990_wp, 1.9253832_wp, 1.3966367_wp, &
      & 2.3227981_wp, 1.5682330_wp, 1.5655420_wp, &
      & 2.3709891_wp, 1.8409795_wp, 1.7423898_wp, &
      & 1.4465300_wp, 1.2040440_wp, 0.0000000_wp, &
      & 1.5059368_wp, 1.3063387_wp, 1.0909195_wp, &
      & 1.8047369_wp, 1.7982376_wp, 1.0955633_wp, &
      & 2.0023112_wp, 1.9021607_wp, 1.3993110_wp, &
      & 2.2563431_wp, 2.1483129_wp, 1.8766631_wp, &
      & 2.5042181_wp, 2.1106542_wp, 1.9511992_wp, &
      & 2.7659636_wp, 2.1692369_wp, 1.8254810_wp, &
      & 0.6993851_wp, 0.8209591_wp, 0.0000000_wp, &
      & 1.1429377_wp, 0.8089888_wp, 1.5500000_wp, &
      & 1.7000000_wp, 1.2510491_wp, 1.0686497_wp, &
      & 1.7925035_wp, 1.5123924_wp, 1.1495969_wp, &
      & 1.7107755_wp, 1.3747235_wp, 0.7765805_wp, &
      & 1.7502703_wp, 1.5772858_wp, 1.4297760_wp, &
      & 1.9777411_wp, 2.0982690_wp, 1.4810776_wp, &
      & 2.0243462_wp, 2.3480304_wp, 1.6149520_wp, &
      & 2.3713759_wp, 2.1418555_wp, 1.5716934_wp, &
      & 2.3655400_wp, 2.0465727_wp, 1.7370737_wp, &
      & 3.1103872_wp, 1.7772461_wp, 1.2958832_wp, &
      & 1.7646331_wp, 1.3127608_wp, 0.0000000_wp, &
      & 1.7863485_wp, 1.4997607_wp, 1.1650646_wp, &
      & 2.2482408_wp, 1.8768725_wp, 1.1662079_wp, &
      & 2.5053498_wp, 1.9880208_wp, 1.7329511_wp, &
      & 2.5955339_wp, 2.1294865_wp, 1.8137622_wp, &
      & 2.1518600_wp, 2.1826296_wp, 1.7939636_wp, &
      & 3.1987630_wp, 2.2995227_wp, 1.7893151_wp, &
      & 0.7514304_wp, 0.6791606_wp, 0.0000000_wp, &
      & 1.2995061_wp, 0.7121741_wp, 1.6000000_wp, &
      & 1.9000000_wp, 1.4262742_wp, 1.3072774_wp, &
      & 1.8651798_wp, 1.5610733_wp, 1.2837815_wp, &
      & 1.8755508_wp, 1.5679201_wp, 1.2726347_wp, &
      & 1.8859217_wp, 1.5747669_wp, 1.2614880_wp, &
      & 1.8962926_wp, 1.5816137_wp, 1.2503412_wp, &
      & 1.9066636_wp, 1.5884605_wp, 1.2391944_wp, &
      & 1.9170345_wp, 1.5953073_wp, 1.2280476_wp, &
      & 1.9274054_wp, 1.6021541_wp, 1.2169008_wp, &
      & 1.9377764_wp, 1.6090009_wp, 1.2057541_wp, &
      & 1.9481473_wp, 1.6158477_wp, 1.1946073_wp, &
      & 1.9585183_wp, 1.6226945_wp, 1.1834605_wp, &
      & 1.9688892_wp, 1.6295413_wp, 1.1723137_wp, &
      & 1.9792601_wp, 1.6363881_wp, 1.1611670_wp, &
      & 1.9896311_wp, 1.6432350_wp, 1.1500202_wp, &
      & 2.0000020_wp, 1.6500818_wp, 1.1388734_wp, &
      & 2.0923623_wp, 1.6165437_wp, 1.2872624_wp, &
      & 1.7483083_wp, 1.6551025_wp, 1.0233550_wp, &
      & 2.0080568_wp, 2.1084082_wp, 1.7641136_wp, &
      & 2.3094018_wp, 1.9812974_wp, 1.7312087_wp, &
      & 2.3795997_wp, 2.2387289_wp, 1.9527035_wp, &
      & 2.5582242_wp, 2.5114785_wp, 1.8368726_wp, &
      & 2.7100253_wp, 2.3315088_wp, 1.8622968_wp, &
      & 3.1211282_wp, 2.3912459_wp, 1.9835079_wp, &
      & 2.1719281_wp, 1.5852934_wp, 0.0000000_wp, &
      & 2.5798532_wp, 1.8750064_wp, 0.0000000_wp, &
      & 2.8266565_wp, 1.9846021_wp, 0.0000000_wp, &
      & 3.0792898_wp, 2.1104258_wp, 0.0000000_wp, &
      & 3.2655001_wp, 2.4581428_wp, 0.0000000_wp, &
      & 2.3566483_wp, 2.4476546_wp, 1.7730006_wp, &
      & 3.5667425_wp, 2.6010762_wp, 1.9384224_wp],&
      & shape(slaterExponent))

   real(wp), parameter :: kQShell(0:2, 1:maxElem) = reshape([&
      & 0.2591008_wp, 0.0000000_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.0526662_wp,-1.7836600_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.9636068_wp, 0.1933068_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.8310555_wp, 0.8910197_wp, 0.0000000_wp, 0.0000000_wp, &
      & 3.1865592_wp, 0.9002623_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1683176_wp, 0.2988821_wp, 0.0000000_wp, 0.0000000_wp, &
      & 1.6770083_wp,-0.2856884_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.5947188_wp, 0.0916715_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.8966240_wp, 0.2645049_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1394174_wp, 0.1181016_wp,-0.1235510_wp, 0.0000000_wp, &
      & 1.4578927_wp,-2.1216651_wp, 0.0000000_wp, 0.0000000_wp, &
      &-1.1457069_wp, 0.6192238_wp,-1.7052287_wp, 0.0000000_wp, &
      & 1.1373983_wp,-0.5030043_wp,-0.2180697_wp, 0.0000000_wp, &
      &-1.6833805_wp, 0.7676860_wp,-0.3164855_wp, 0.0000000_wp, &
      & 0.0638088_wp,-0.0488732_wp, 0.1144146_wp, 0.0000000_wp, &
      &-0.3628979_wp,-0.1162502_wp, 0.5229391_wp, 0.0000000_wp, &
      &-0.1771804_wp, 0.3541391_wp, 0.1082906_wp, 0.0000000_wp, &
      & 0.0581999_wp, 0.1172583_wp, 0.2579057_wp, 0.0000000_wp, &
      & 1.4171788_wp, 0.6090963_wp, 0.0000000_wp, 0.0000000_wp, &
      &-1.3567908_wp, 0.7982482_wp,-2.4352495_wp, 0.0000000_wp, &
      &-1.2515726_wp,-0.2288979_wp,-0.1126341_wp, 0.0000000_wp, &
      &-2.6329946_wp,-0.5589503_wp,-0.6584541_wp, 0.0000000_wp, &
      &-2.8582869_wp,-0.9497022_wp,-0.1743943_wp, 0.0000000_wp, &
      &-0.6557652_wp,-1.1133287_wp, 0.5864026_wp, 0.0000000_wp, &
      &-2.1012472_wp, 0.0931727_wp,-0.2296589_wp, 0.0000000_wp, &
      &-2.1531278_wp,-1.3014729_wp,-0.2054193_wp, 0.0000000_wp, &
      &-1.6422305_wp,-0.2413613_wp, 0.1032325_wp, 0.0000000_wp, &
      & 0.0892590_wp, 0.6269495_wp,-0.0372394_wp, 0.0000000_wp, &
      & 0.0790442_wp, 0.0667641_wp,-0.2727800_wp, 0.0000000_wp, &
      & 0.5947695_wp,-0.3462432_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.8467361_wp,-0.2584210_wp,-0.5108585_wp, 0.0000000_wp, &
      & 0.2348166_wp,-0.2410370_wp,-0.2772269_wp, 0.0000000_wp, &
      & 1.1862714_wp, 1.3281349_wp, 0.6150681_wp, 0.0000000_wp, &
      & 0.1524644_wp, 0.6961378_wp,-1.0725308_wp, 0.0000000_wp, &
      &-0.0337658_wp, 0.3715817_wp,-0.6430662_wp, 0.0000000_wp, &
      &-0.0658049_wp, 0.0634599_wp, 1.3725483_wp, 0.0000000_wp, &
      &-2.1110486_wp,-1.3455843_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.7184537_wp, 0.0777799_wp,-0.8135551_wp, 0.0000000_wp, &
      & 1.1237295_wp,-0.9370621_wp, 0.1974924_wp, 0.0000000_wp, &
      &-3.1290732_wp,-0.6950275_wp, 0.2953364_wp, 0.0000000_wp, &
      &-1.5564962_wp,-0.4070963_wp,-0.0034815_wp, 0.0000000_wp, &
      &-0.5021863_wp,-2.0445824_wp, 0.4028728_wp, 0.0000000_wp, &
      &-3.2915311_wp,-1.3215133_wp,-0.0229317_wp, 0.0000000_wp, &
      &-1.9247370_wp,-3.0950796_wp, 0.0242021_wp, 0.0000000_wp, &
      &-1.6523781_wp,-0.2966759_wp,-0.2394570_wp, 0.0000000_wp, &
      &-2.2410456_wp,-1.0409758_wp,-0.5186905_wp, 0.0000000_wp, &
      &-2.6933989_wp, 1.6377067_wp, 0.4579394_wp, 0.0000000_wp, &
      & 0.6273704_wp,-0.1339630_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.6476509_wp,-0.3983757_wp,-0.5489296_wp, 0.0000000_wp, &
      & 0.3428358_wp,-0.0107237_wp,-0.0368582_wp, 0.0000000_wp, &
      & 0.2761934_wp, 0.3213207_wp, 0.8605258_wp, 0.0000000_wp, &
      &-0.6640266_wp, 0.6301827_wp,-1.2044909_wp, 0.0000000_wp, &
      & 0.1264617_wp, 0.1662148_wp, 0.0032647_wp, 0.0000000_wp, &
      &-0.1330582_wp, 0.0494917_wp, 2.8529059_wp, 0.0000000_wp, &
      &-0.9632540_wp,-0.3851189_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.9508608_wp,-0.3461083_wp,-1.0044558_wp, 0.0000000_wp, &
      & 0.8353349_wp,-0.6965739_wp, 0.1247060_wp, 0.0000000_wp, &
      & 0.2963503_wp,-0.5819361_wp, 0.2692711_wp, 0.0000000_wp, &
      & 0.3049667_wp,-0.5242933_wp, 0.2546408_wp, 0.0000000_wp, &
      & 0.3135831_wp,-0.4666506_wp, 0.2400105_wp, 0.0000000_wp, &
      & 0.3221996_wp,-0.4090079_wp, 0.2253803_wp, 0.0000000_wp, &
      & 0.3308160_wp,-0.3513652_wp, 0.2107500_wp, 0.0000000_wp, &
      & 0.3394324_wp,-0.2937224_wp, 0.1961197_wp, 0.0000000_wp, &
      & 0.3480488_wp,-0.2360797_wp, 0.1814894_wp, 0.0000000_wp, &
      & 0.3566653_wp,-0.1784370_wp, 0.1668591_wp, 0.0000000_wp, &
      & 0.3652817_wp,-0.1207942_wp, 0.1522288_wp, 0.0000000_wp, &
      & 0.3738981_wp,-0.0631515_wp, 0.1375985_wp, 0.0000000_wp, &
      & 0.3825146_wp,-0.0055088_wp, 0.1229683_wp, 0.0000000_wp, &
      & 0.3911310_wp, 0.0521340_wp, 0.1083380_wp, 0.0000000_wp, &
      & 0.3997474_wp, 0.1097767_wp, 0.0937077_wp, 0.0000000_wp, &
      & 0.4083638_wp, 0.1674194_wp, 0.0790774_wp, 0.0000000_wp, &
      & 0.2914598_wp, 0.2000813_wp, 0.1081073_wp, 0.0000000_wp, &
      &-1.8120352_wp, 1.5015902_wp,-0.0324750_wp, 0.0000000_wp, &
      &-1.5162303_wp,-0.5627344_wp,-0.0063703_wp, 0.0000000_wp, &
      &-0.9565221_wp,-0.4840866_wp,-0.7237461_wp, 0.0000000_wp, &
      &-1.7054596_wp,-0.5268454_wp,-0.2408914_wp, 0.0000000_wp, &
      &-1.8847069_wp,-0.4888169_wp,-0.1606430_wp, 0.0000000_wp, &
      &-1.7277302_wp,-1.6224161_wp,-0.4690687_wp, 0.0000000_wp, &
      &-3.8714610_wp, 1.0468549_wp, 0.5645830_wp, 0.0000000_wp, &
      & 1.2916136_wp,-0.3520016_wp, 0.0000000_wp, 0.0000000_wp, &
      & 2.7768835_wp,-1.1939420_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.1764125_wp, 0.6143160_wp, 0.0000000_wp, 0.0000000_wp, &
      & 0.1602211_wp, 0.4235069_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.1211837_wp, 0.6131814_wp, 0.0000000_wp, 0.0000000_wp, &
      &-0.7035045_wp, 0.1660577_wp, 0.6738787_wp, 0.0000000_wp, &
      & 0.2212935_wp, 0.0793110_wp, 1.3169147_wp, 0.0000000_wp],&
      & shape(kQShell))

   real(wp), parameter :: kQAtom(1:maxElem) = [&
      & 0.2473983_wp,-0.7787934_wp,-2.0181966_wp, 0.2010595_wp, 0.5604651_wp, &
      & 0.0908886_wp,-0.0630602_wp,-0.0504985_wp, 0.0307638_wp,-0.4585161_wp, &
      &-1.1795902_wp,-0.5383484_wp,-0.2891414_wp,-0.2539732_wp,-0.4636669_wp, &
      & 0.0175812_wp,-0.4977269_wp,-1.8034383_wp,-0.2695403_wp, 0.2084025_wp, &
      &-0.6961569_wp,-0.0444615_wp,-0.1157893_wp,-0.1506473_wp,-0.0193450_wp, &
      & 0.1358351_wp,-0.0258463_wp,-0.1389803_wp, 0.1832504_wp, 0.2795320_wp, &
      &-0.2653902_wp,-0.3850791_wp,-0.4048451_wp,-0.0331375_wp,-0.3771185_wp, &
      &-0.1964130_wp, 1.5782426_wp, 0.1783732_wp,-0.5462958_wp,-0.1563616_wp, &
      & 0.2100641_wp, 0.5064106_wp, 0.0670438_wp, 0.2937971_wp, 0.0788835_wp, &
      & 0.4874039_wp,-0.4078822_wp, 0.2261002_wp,-0.4403034_wp,-0.2304807_wp, &
      &-0.3614513_wp,-0.0339759_wp,-0.1031505_wp,-0.1821550_wp, 0.9414952_wp, &
      & 0.7618755_wp,-0.7252284_wp,-0.8403891_wp,-0.8014986_wp,-0.7626081_wp, &
      &-0.7237176_wp,-0.6848271_wp,-0.6459366_wp,-0.6070461_wp,-0.5681557_wp, &
      &-0.5292652_wp,-0.4903747_wp,-0.4514842_wp,-0.4125937_wp,-0.3737032_wp, &
      &-0.3348127_wp,-0.1572535_wp, 0.0625598_wp,-0.1822002_wp,-0.0546542_wp, &
      & 0.3585959_wp, 0.0403085_wp, 0.4160851_wp, 0.5364732_wp, 0.1234906_wp, &
      &-0.5902190_wp, 0.0731828_wp, 0.1795929_wp,-0.0924653_wp,-0.3033414_wp, &
      &-0.0946010_wp]


contains


subroutine initData(self)

   !> Data instance
   type(TxTBData), intent(out) :: self

   self%name = 'GFN0-xTB'
   self%level = 0
   self%nShell = nShell(:maxElem)

   call initGFN0(self%repulsion)
   call initGFN0(self%coulomb, self%nShell)
   call initGFN0(self%hamiltonian, self%nShell)

end subroutine initData


subroutine initRepulsion(self)

   !> Data instance
   type(TRepulsionData), intent(out) :: self

   call init(self, kExp, kExpLight, rExp, 0.0_wp, repAlpha, repZeff, &
      & electronegativity)

end subroutine initRepulsion


subroutine initCoulomb(self, nShell)

   !> Data instance
   type(TCoulombData), intent(out) :: self

   !>
   integer, intent(in) :: nShell(:)

   self%electronegativity = eeqChi(1:maxElem)
   self%chemicalHardness = eeqGam(1:maxElem)
   self%kCN = eeqkCN(1:maxElem)
   self%chargeWidth = eeqAlp(1:maxElem)

end subroutine initCoulomb


subroutine initHamiltonian(self, nShell)

   !> Data instance
   type(THamiltonianData), intent(out) :: self

   !>
   integer, intent(in) :: nShell(:)

   integer :: mShell, nPrim, lAng
   integer :: iZp, iSh
   logical :: valShell(0:3)

   mShell = maxval(nShell)
   self%angShell = angShell(:mShell, :maxElem)

   self%kScale = 0.5_wp*(spread(gfn0Globals%kShell,1,4)+spread(gfn0Globals%kShell,2,4))
   self%kDiff = gfn0Globals%kDiff
   self%enScale = 0.005_wp * (spread(gfn0Globals%enshell, 1, 4) &
      & + spread(gfn0Globals%enshell, 2, 4))
   self%enScale4 = gfn0Globals%enscale4

   self%electronegativity = electronegativity(:maxElem)
   self%atomicRad = atomicRad(:maxElem)
   self%shellPoly = shellPoly(:, :maxElem)
   self%selfEnergy = selfEnergy(:mShell, :maxElem)
   self%slaterExponent = slaterExponent(:mShell, :maxElem)
   self%principalQuantumNumber = principalQuantumNumber(:mShell, :maxElem)
   self%kQAtom = kQAtom(:maxElem)

   allocate(self%kCN(mShell, maxElem))
   call angToShellData(self%kCN, nShell, self%angShell, kCN)

   allocate(self%kQShell(mShell, maxElem))
   call angToShellData(self%kQShell, nShell, self%angShell, kQShell)

   allocate(self%pairParam(maxElem, maxElem))
   call setGFN0PairParam(self%pairParam)

   allocate(self%valenceShell(mShell, maxElem))
   call generateValenceShellData(self%valenceShell, nShell, self%angShell)

   allocate(self%referenceOcc(mShell, maxElem))
   call setGFN2ReferenceOcc(self, nShell)

   allocate(self%numberOfPrimitives(mShell, maxElem))
   call setGFN0NumberOfPrimitives(self, nShell)

end subroutine initHamiltonian


!>
subroutine setGFN0NumberOfPrimitives(self, nShell)

   !> Data instance
   type(THamiltonianData), intent(inout) :: self

   !>
   integer, intent(in) :: nShell(:)

   integer :: nPrim, iZp, iSh

   do iZp = 1, maxElem
      do iSh = 1, nShell(iZp)
         nPrim = 0
         if (iZp <= 2) then
            select case(self%angShell(iSh, iZp))
            case(0)
               if (self%valenceShell(iSh, iZp) /= 0) then
                  nPrim = 3
               else
                  nPrim = 2
               end if
            case(1)
               nPrim = 3
            end select
         else
            select case(self%angShell(iSh, iZp))
            case(0)
               if (self%principalQuantumNumber(iSh, iZp) > 5) then
                  nPrim = 6
               else
                  nPrim = 4
               end if
            case(1)
               if (self%principalQuantumNumber(iSh, iZp) > 5) then
                  nPrim = 6
               else
                  nPrim = 3
               end if
            case(2, 3)
               nPrim = 4
            end select
         end if
         self%numberOfPrimitives(iSh, iZp) = nPrim
      end do
   end do

end subroutine setGFN0NumberOfPrimitives


subroutine setGFN0PairParam(pairParam)

   real(wp), allocatable :: pairParam(:, :)

   integer :: iZp, jZp
   logical :: iTm, jTm, iCm, jCm

   pairParam(:, :) = 1.0_wp

   do iZp = 21, 79
      do jZp = 21, iZp
         iTm = dBlock(iZp)
         jTm = dBlock(jZp)
         if (iTm .and. jTm) then
            iCm = coinMetal(iZp)
            jCm = coinMetal(jZp)
            if (iCm .and. jCm) then
               pairParam(iZp, jZp) = 0.9_wp
               pairParam(jZp, iZp) = 0.9_wp
            else
               pairParam(iZp, jZp) = 1.1_wp
               pairParam(jZp, iZp) = 1.1_wp
            end if
         end if
      end do
   end do

contains

elemental function coinMetal(zp) result(cm)
   integer, intent(in) :: zp
   logical :: cm

   cm = any(zp == [29, 47, 79])

end function coinMetal

elemental function dBlock(zp) result(tm)
   integer, intent(in) :: zp
   logical :: tm

   tm = zp > 20 .and. zp < 30 .or. zp > 38 .and. zp < 48 .or. zp > 56 .and. zp < 80

end function dBlock

end subroutine setGFN0PairParam


end module xtb_xtb_gfn0
