/*
 * @(#)PyraminxU.c
 *
 * Copyright 1994 - 2023  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Undo algorithm for Pyraminx */

#include "PyraminxP.h"

PyraminxLoc *startLoc[MAX_FACES] =
{
	NULL, NULL, NULL, NULL
};

static void
newStack(PyraminxStack *s)
{
	if (s->lastMove != NULL || s->firstMove != NULL)
		return;
	if (!(s->lastMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (!(s->firstMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->firstMove->previous = s->lastMove->next = NULL;
	s->firstMove->next = s->lastMove;
	s->lastMove->previous = s->firstMove;
	s->count = 0;
}

static void
pushStack(PyraminxStack *s, moveRecord **move)
{
	if (!(s->currMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->lastMove->previous->next = s->currMove;
	s->currMove->previous = s->lastMove->previous;
	s->currMove->next = s->lastMove;
	s->lastMove->previous = s->currMove;
	*move = &(s->currMove->move);
	s->count++;
}

static void
popStack(PyraminxStack *s)
{
	s->currMove = s->lastMove->previous;
	s->lastMove->previous->previous->next = s->lastMove;
	s->lastMove->previous = s->lastMove->previous->previous;
	free(s->currMove);
	s->count--;
}

static moveRecord *
topStack(PyraminxStack *s)
{
	return &(s->lastMove->previous->move);
}

static int
emptyStack(PyraminxStack *s)
{
	return (s->lastMove->previous == s->firstMove);
}

static void
flushStack(PyraminxStack *s)
{
	while (s->lastMove->previous != s->firstMove) {
		s->currMove = s->lastMove->previous;
		s->lastMove->previous->previous->next = s->lastMove;
		s->lastMove->previous = s->lastMove->previous->previous;
		free(s->currMove);
	}
	s->count = 0;
}

static void
deleteStack(PyraminxStack *s)
{
	flushStack(s);
	if (s->firstMove) {
		free(s->firstMove);
		s->firstMove = NULL;
	}
	if (s->lastMove) {
		free(s->lastMove);
		s->lastMove = NULL;
	}
}

/**********************************/

void
newMoves(PyraminxStack *s)
{
	newStack(s);
}

void
deleteMoves(PyraminxStack *s)
{
	deleteStack(s);
}

static void
writeMove(moveRecord *move, int direction, int style, int control,
		int face, int position)
{
#if 0
	move->direction = direction;
	move->style = style;
	move->control = control;
	move->face = face;
#endif
	move->packed = (unsigned short int) (((control & 0xF) << 12) +
		((style & 0xF) << 8) + ((direction & 0xF) << 4) +
		(face & 0xF));
	move->position = position;
}

static void
readMove(moveRecord *move, int *direction, int *style, int *control,
		int *face, int *position)
{
#if 0
	*direction = move->direction;
	*style = move->style;
	*control = move->control;
	*face = move->face;
#endif
	*face = (int) (move->packed & 0xF);
	*direction = (int) ((move->packed >> 4) & 0xF);
	*style = (int) ((move->packed >> 8) & 0xF);
	*control = (int) ((move->packed >> 12) & 0xF);
	*position = move->position;
}

void
setMove(PyraminxStack *s, int direction, int style, int control,
		int face, int position)
{
	moveRecord *move;

	pushStack(s, &move);
	writeMove(move, direction, style, control, face, position);
}

void
getMove(PyraminxStack *s, int *direction, int *style, int *control,
		int *face, int *position)
{
	readMove(topStack(s), direction, style, control, face, position);
	popStack(s);
}

int
madeMoves(PyraminxStack *s)
{
	return !emptyStack(s);
}

void
flushMoves(PyraminxWidget w, PyraminxStack *s, Boolean undo)
{
	int face, position;

	flushStack(s);
	if (undo) {
		for (face = 0; face < MAX_FACES; face++)
			for (position = 0; position < w->pyraminx.sizeSize; position++) {
				startLoc[face][position].face =
					w->pyraminx.facetLoc[face][position].face;
				startLoc[face][position].rotation =
					w->pyraminx.facetLoc[face][position].rotation;
			}
	}
}

int
numMoves(PyraminxStack *s)
{
	return s->count;
}

Boolean
scanMoves(FILE *fp, PyraminxWidget w, int moves)
{
	int direction, style, control, face, position, k, c;

	for (k = 0; k < moves; k++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		if (fscanf(fp, "%d %d %d %d %d",
				&direction, &style, &control, &face, &position) != 5) {
			(void) fprintf(stderr,
				"corrupt scan, expecting 5 integers for move %d\n", k);
			return False;
		}
		movePuzzle(w, face, position, direction,
			style, control, INSTANT);
	}
	return True;
}

void
printMoves(FILE *fp, PyraminxStack *s)
{
	int direction, style, control, face, position, counter = 0;

	s->currMove = s->firstMove->next;
	(void) fprintf(fp, "moves\tdir\tstyle\tcontrol\tface\tposition\n");
	while (s->currMove != s->lastMove) {
		readMove(&(s->currMove->move),
			&direction, &style, &control, &face, &position);
		(void) fprintf(fp, "%d%c\t%d\t%d\t%d\t%d\t%d\n",
		++counter, SYMBOL, direction, style, control, face, position);
		s->currMove = s->currMove->next;
	}
}

Boolean
scanStartPosition(FILE *fp, PyraminxWidget w)
{
	int face, position, num, c;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			if (fscanf(fp, "%d ", &num) != 1) {
				(void) fprintf(stderr,
					"corrupt start record, expecting an integer for %d %d\n",
					face, position);
				return False;
			}
			startLoc[face][position].face = num;
			if (w->pyraminx.orient) {
				if (fscanf(fp, "%d ", &num) != 1) {
					(void) fprintf(stderr,
						"corrupt start record, expecting an integer for orient %d %d\n",
						face, position);
					return False;
				}
				startLoc[face][position].rotation = num;
			}
		}
	return True;
}

void
printStartPosition(FILE *fp, PyraminxWidget w)
{
	int face, position;

	(void) fprintf(fp, "\nstartingPosition%c\n", SYMBOL);
	for (face = 0; face < MAX_FACES; face++) {
		int square_1 = 0, sqrt = 1;
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			(void) fprintf(fp, "%4d",
				startLoc[face][position].face);
			if (w->pyraminx.orient) {
				(void) fprintf(fp, "%3d",
					startLoc[face][position].rotation);
			}
			if (position == square_1) {
				(void) fprintf(fp, "\n");
				++sqrt;
				square_1 = sqrt * sqrt - 1;
			}
		}
		(void) fprintf(fp, "\n");
	}
}

void
setStartPosition(PyraminxWidget w)
{
	int face, position;

	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			w->pyraminx.facetLoc[face][position].face =
				startLoc[face][position].face;
			if (w->pyraminx.orient) {
				w->pyraminx.facetLoc[face][position].rotation =
					startLoc[face][position].rotation;
			}
		}
	drawAllPieces(w);
}
